import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import { exec } from 'child_process';
import { promisify } from 'util';

const execAsync = promisify(exec);

export async function POST(request: NextRequest) {
    try {
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').split('T')[0] + '-' +
            new Date().toTimeString().split(' ')[0].replace(/:/g, '');
        const backupName = `domain-manager-full-backup-${timestamp}`;
        const backupDir = path.join(process.cwd(), '..', 'backups', backupName);
        const backupArchive = `${backupDir}.tar.gz`;

        console.log('🔄 Creating full backup:', backupName);

        // יצירת תיקיית גיבוי
        fs.mkdirSync(backupDir, { recursive: true });

        // העתקת קבצים (למעט node_modules, .next, .git)
        const excludePatterns = [
            'node_modules',
            '.next',
            '.git',
            '*.log',
            '.DS_Store',
            '*.tmp'
        ];

        console.log('📁 Copying files...');

        // העתקת כל הקבצים
        const copyCommand = `rsync -av --progress ${excludePatterns.map(p => `--exclude='${p}'`).join(' ')} . "${backupDir}/"`;
        await execAsync(copyCommand);

        // העתקת בסיסי נתונים
        console.log('💾 Copying databases...');
        const dbFiles = fs.readdirSync(process.cwd()).filter(file => file.endsWith('.db'));
        dbFiles.forEach(dbFile => {
            fs.copyFileSync(path.join(process.cwd(), dbFile), path.join(backupDir, dbFile));
        });

        // יצירת ארכיון
        console.log('📦 Creating archive...');
        const archiveCommand = `cd "${path.dirname(backupDir)}" && tar -czf "${path.basename(backupArchive)}" "${path.basename(backupDir)}"`;
        await execAsync(archiveCommand);

        // מחיקת תיקיית הגיבוי (נשאר רק הארכיון)
        fs.rmSync(backupDir, { recursive: true, force: true });

        // קבלת גודל הארכיון
        const stats = fs.statSync(backupArchive);
        const size = formatBytes(stats.size);

        console.log('✅ Backup completed:', backupName);

        return NextResponse.json({
            success: true,
            backupName,
            backupPath: backupArchive,
            size,
            timestamp: new Date().toISOString(),
            message: 'גיבוי מלא נוצר בהצלחה'
        });

    } catch (error: any) {
        console.error('❌ Backup creation failed:', error);
        return NextResponse.json({
            error: 'שגיאה ביצירת הגיבוי: ' + error.message
        }, { status: 500 });
    }
}

function formatBytes(bytes: number): string {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}
