import { NextRequest, NextResponse } from 'next/server';
import { validatePasswordResetToken, updatePassword, deletePasswordResetToken } from '@/src/lib/db-auth';
import { validatePassword } from '@/src/lib/auth';
import { initAuthDatabase } from '@/src/lib/db-auth';

export async function POST(request: NextRequest) {
    try {
        // Initialize auth database
        await initAuthDatabase();

        const { token, newPassword } = await request.json();

        if (!token || !newPassword) {
            return NextResponse.json(
                { error: 'טוקן וסיסמה חדשה נדרשים' },
                { status: 400 }
            );
        }

        // Validate token
        const tokenData = await validatePasswordResetToken(token);
        if (!tokenData) {
            return NextResponse.json(
                { error: 'קישור איפוס לא תקין או פג תוקף' },
                { status: 400 }
            );
        }

        // Validate password strength
        const passwordValidation = validatePassword(newPassword);
        if (!passwordValidation.isValid) {
            return NextResponse.json(
                { error: passwordValidation.errors.join(', ') },
                { status: 400 }
            );
        }

        // Update password
        await updatePassword(tokenData.userId, newPassword);

        // Delete used token
        await deletePasswordResetToken(token);

        return NextResponse.json({
            message: 'הסיסמה אופסה בהצלחה',
        });
    } catch (error) {
        console.error('Reset password error:', error);
        return NextResponse.json(
            { error: 'שגיאה פנימית בשרת' },
            { status: 500 }
        );
    }
}
