import { NextRequest, NextResponse } from 'next/server';
import { createUser } from '@/src/lib/db-auth';
import { generateToken, setAuthCookie, validatePassword } from '@/src/lib/auth';
import { initAuthDatabase } from '@/src/lib/db-auth';

export async function POST(request: NextRequest) {
    try {
        // Initialize auth database
        await initAuthDatabase();

        const { email, password, name, role } = await request.json();

        if (!email || !password || !name) {
            return NextResponse.json(
                { error: 'כל השדות נדרשים' },
                { status: 400 }
            );
        }

        // Validate password strength
        const passwordValidation = validatePassword(password);
        if (!passwordValidation.isValid) {
            return NextResponse.json(
                { error: passwordValidation.errors.join(', ') },
                { status: 400 }
            );
        }

        // Validate email format
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(email)) {
            return NextResponse.json(
                { error: 'כתובת אימייל לא תקינה' },
                { status: 400 }
            );
        }

        // Validate name
        if (name.trim().length < 2) {
            return NextResponse.json(
                { error: 'השם חייב להכיל לפחות 2 תווים' },
                { status: 400 }
            );
        }

        // Create user
        const user = await createUser({
            email: email.toLowerCase().trim(),
            password,
            name: name.trim(),
            role: role || 'USER',
        });

        // Generate JWT token
        const token = generateToken({
            userId: user.id,
            email: user.email,
            role: user.role,
        });

        // Create response
        const response = NextResponse.json({
            user: {
                id: user.id,
                email: user.email,
                name: user.name,
                role: user.role,
                isActive: user.isActive,
                createdAt: user.createdAt,
                updatedAt: user.updatedAt,
            },
            token,
        });

        // Set auth cookie
        setAuthCookie(token, response);

        return response;
    } catch (error: any) {
        console.error('Registration error:', error);

        // Handle unique constraint violation
        if (error.code === 'SQLITE_CONSTRAINT_UNIQUE') {
            return NextResponse.json(
                { error: 'כתובת אימייל כבר קיימת במערכת' },
                { status: 409 }
            );
        }

        return NextResponse.json(
            { error: 'שגיאה פנימית בשרת' },
            { status: 500 }
        );
    }
}
