import { NextRequest, NextResponse } from 'next/server';
import bcrypt from 'bcryptjs';
import { findUserByEmail, updateLastLogin } from '@/src/lib/db-auth';
import { generateToken, setAuthCookie } from '@/src/lib/auth';
import { initAuthDatabase } from '@/src/lib/db-auth';

export async function POST(request: NextRequest) {
    try {
        // Initialize auth database
        await initAuthDatabase();

        const { email, password } = await request.json();

        if (!email || !password) {
            return NextResponse.json(
                { error: 'אימייל וסיסמה נדרשים' },
                { status: 400 }
            );
        }

        // Find user by email
        const user = await findUserByEmail(email);
        if (!user) {
            return NextResponse.json(
                { error: 'פרטי התחברות שגויים' },
                { status: 401 }
            );
        }

        // Verify password
        const isPasswordValid = await bcrypt.compare(password, user.password);
        if (!isPasswordValid) {
            return NextResponse.json(
                { error: 'פרטי התחברות שגויים' },
                { status: 401 }
            );
        }

        // Check if user is active
        if (!user.isActive) {
            return NextResponse.json(
                { error: 'החשבון לא פעיל' },
                { status: 401 }
            );
        }

        // Update last login
        await updateLastLogin(user.id);

        // Generate JWT token
        const token = await generateToken({
            userId: user.id,
            email: user.email,
            role: user.role,
        });

        // Create response
        const response = NextResponse.json({
            user: {
                id: user.id,
                email: user.email,
                name: user.name,
                role: user.role,
                isActive: user.isActive,
                createdAt: user.createdAt,
                updatedAt: user.updatedAt,
                lastLoginAt: new Date().toISOString(),
            },
            token,
        });

        // Set auth cookie
        setAuthCookie(token, response);

        return response;
    } catch (error) {
        console.error('Login error:', error);
        return NextResponse.json(
            { error: 'שגיאה פנימית בשרת' },
            { status: 500 }
        );
    }
}
