import { NextRequest, NextResponse } from 'next/server';
import { findUserByEmail, createPasswordResetToken } from '@/src/lib/db-auth';
import { generateResetToken } from '@/src/lib/auth';
import { initAuthDatabase } from '@/src/lib/db-auth';

export async function POST(request: NextRequest) {
    try {
        // Initialize auth database
        await initAuthDatabase();

        const { email } = await request.json();

        if (!email) {
            return NextResponse.json(
                { error: 'כתובת אימייל נדרשת' },
                { status: 400 }
            );
        }

        // Find user by email
        const user = await findUserByEmail(email);
        if (!user) {
            // Don't reveal if user exists or not for security
            return NextResponse.json({
                message: 'אם כתובת האימייל קיימת במערכת, נשלח לך קישור איפוס סיסמה',
            });
        }

        // Generate reset token
        const resetToken = generateResetToken();

        // Save reset token to database
        await createPasswordResetToken(user.id, resetToken);

        // TODO: Send email with reset link
        // For now, we'll just log the token (in production, send email)
        console.log(`Reset token for ${user.email}: ${resetToken}`);
        console.log(`Reset URL: ${process.env.NEXTAUTH_URL || 'http://localhost:3000'}/reset-password?token=${resetToken}`);

        return NextResponse.json({
            message: 'אם כתובת האימייל קיימת במערכת, נשלח לך קישור איפוס סיסמה',
        });
    } catch (error) {
        console.error('Forgot password error:', error);
        return NextResponse.json(
            { error: 'שגיאה פנימית בשרת' },
            { status: 500 }
        );
    }
}
