import { NextRequest, NextResponse } from 'next/server';
import { openDB } from '@/src/lib/db';
import { writeFile, mkdir } from 'fs/promises';
import path from 'path';
import { existsSync } from 'fs';

export async function POST(request: NextRequest) {
    try {
        const formData = await request.formData();
        const file = formData.get('file') as File;
        const entityType = formData.get('entity_type') as string;
        const entityId = formData.get('entity_id') as string;
        const isVoiceRecording = formData.get('is_voice_recording') === 'true';
        const duration = formData.get('duration') ? parseInt(formData.get('duration') as string) : null;
        const notes = formData.get('notes') as string | null;

        if (!file || !entityType || !entityId) {
            return NextResponse.json(
                { success: false, error: 'Missing required fields' },
                { status: 400 }
            );
        }

        // Generate unique filename
        const timestamp = Date.now();
        const originalName = file.name;
        const extension = path.extname(originalName);
        const fileName = `${timestamp}${extension}`;

        // Create directory path
        const uploadDir = path.join(process.cwd(), 'public', 'uploads', entityType, entityId);

        // Ensure directory exists
        if (!existsSync(uploadDir)) {
            await mkdir(uploadDir, { recursive: true });
        }

        // Save file
        const filePath = path.join(uploadDir, fileName);
        const bytes = await file.arrayBuffer();
        const buffer = Buffer.from(bytes);
        await writeFile(filePath, buffer);

        // Save to database
        const db = await openDB();
        const relativePath = `/uploads/${entityType}/${entityId}/${fileName}`;

        return new Promise((resolve) => {
            db.run(
                `INSERT INTO attachments (
          entity_type, entity_id, file_name, original_name, 
          file_type, file_size, mime_type, file_path, 
          is_voice_recording, duration, notes
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                [
                    entityType,
                    parseInt(entityId),
                    fileName,
                    originalName,
                    extension.replace('.', ''),
                    file.size,
                    file.type,
                    relativePath,
                    isVoiceRecording ? 1 : 0,
                    duration,
                    notes
                ],
                function (err) {
                    if (err) {
                        console.error('Error saving attachment to database:', err);
                        resolve(NextResponse.json(
                            { success: false, error: 'Failed to save attachment' },
                            { status: 500 }
                        ));
                    } else {
                        // Fetch the created attachment
                        db.get(
                            'SELECT * FROM attachments WHERE id = ?',
                            [this.lastID],
                            (err, row: unknown) => {
                                if (err) {
                                    resolve(NextResponse.json(
                                        { success: false, error: 'Failed to fetch attachment' },
                                        { status: 500 }
                                    ));
                                } else {
                                    resolve(NextResponse.json({
                                        success: true,
                                        attachment: {
                                            ...row,
                                            is_voice_recording: Boolean(row.is_voice_recording)
                                        }
                                    }));
                                }
                            }
                        );
                    }
                }
            );
        });
    } catch (error) {
        console.error('Error uploading file:', error);
        return NextResponse.json(
            { success: false, error: 'Internal server error' },
            { status: 500 }
        );
    }
}

